﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace VishalMonpara.TokenExample
{
    public interface Token
    {
        /// <summary>
        /// Replaces the token.
        /// </summary>
        /// <param name="input">The input.</param>
        /// <returns></returns>
        string ReplaceToken(string input);
    }

    /// <summary>
    /// Replaces date token with the date value
    /// </summary>
    public abstract class DateToken : Token
    {
        private Regex find = null;

        protected string dateformat = "";

        /// <summary>
        /// Initializes a new instance of the <see cref="DateToken"/> class.
        /// </summary>
        /// <param name="find">The find.</param>
        public DateToken(Regex find)
        {
            this.find = find;
        }
        /// <summary>
        /// Replaces the token.
        /// </summary>
        /// <param name="input">The input.</param>
        /// <returns></returns>
        public string ReplaceToken(string input)
        {

            return find.Replace(input, delegate(Match m)
            {
                return this.GetDateTimeValue().ToString(m.Groups[1].ToString());
            });
        }
        public abstract DateTime GetDateTimeValue();
    }
    /// <summary>
    /// Relaces the counter token with counter value. Counter is increased each time it is called
    /// </summary>
    public class CounterToken:Token
    {
        private Regex find = new Regex("%counter{(.+?)}");
        protected int startValue = 1;
        protected int increment = 1;
        protected string numberformat = "";
        public CounterToken(int startValue, int incrementBy) 
        {
            this.startValue = startValue;
            this.increment = incrementBy;
        }
        public string ReplaceToken(string input)
        {
            return find.Replace(input, delegate(Match m)
            {
                return this.GetNumberValue().ToString(m.Groups[1].ToString());
            });
        }
        public int GetNumberValue()
        {
            int temp = startValue;
            startValue+=increment;
            return temp;
        }
    }
    /// <summary>
    /// Gives the current date time token. Whenever the function is called, value will always be the current date
    /// </summary>
    public class CurrentDateTimeToken : DateToken
    {
        public CurrentDateTimeToken()
            : base(new Regex("%date{(.+?)}"))
        {

        }

        public override DateTime GetDateTimeValue()
        {
            return DateTime.Now;
        }
    }
    /// <summary>
    /// Instantiates the current datetime token but the value will be static during the lifetime of the object
    /// </summary>
    public class FrozenDateTimeToken: DateToken
    {
        DateTime frozen = DateTime.Now;
        public FrozenDateTimeToken(): base(new Regex("%frozendate{(.+?)}"))
        {

        }
        public override DateTime GetDateTimeValue()
        {
            return frozen;
        }
    }
    /// <summary>
    /// Gives yesterday's date time token
    /// </summary>
    public class YesterdayDateTimeToken : DateToken
    {
        public YesterdayDateTimeToken()
            : base(new Regex("%yesterday{(.+?)}"))
        {

        }
        public override DateTime GetDateTimeValue()
        {
            return DateTime.Now.AddDays(-1);
        }
    }
    /// <summary>
    /// Gives last week's date time token
    /// </summary>
    public class LastWeekDateTimeToken : DateToken
    {
        public LastWeekDateTimeToken()
            : base(new Regex("%lastweek{(.+?)}"))
        {

        }
        public override DateTime GetDateTimeValue()
        {
            return DateTime.Now.AddDays(-7);
        }
    }
    /// <summary>
    /// Gives last month's date time token
    /// </summary>
    public class LastMonthDateTimeToken : DateToken
    {
        public LastMonthDateTimeToken()
            : base(new Regex("%lastmonth{(.+?)}"))
        {

        }
        public override DateTime GetDateTimeValue()
        {
            return DateTime.Now.AddMonths(-1);
        }
    }
    /// <summary>
    /// Gives last year's date time token
    /// </summary>
    public class LastYearDateTimeToken : DateToken
    {
        public LastYearDateTimeToken()
            : base(new Regex("%lastyear{(.+?)}"))
        {

        }
        public override DateTime GetDateTimeValue()
        {
            return DateTime.Now.AddYears(-1);
        }
    }
    /// <summary>
    /// Gives user defined date time token. User will specify the token name and value
    /// </summary>
    public class CustomDateTimeToken : DateToken
    {
        DateTime customdatetime = DateTime.MinValue;
        public CustomDateTimeToken(string tokenName, DateTime dateTimeValue)
            : base(new Regex("%" + tokenName + "{(.+?)}"))
        {
            customdatetime = dateTimeValue;
        }
        public override DateTime GetDateTimeValue()
        {
            return customdatetime;
        }
    }
    /// <summary>
    /// Gives the text token. User will specify token name and text
    /// </summary>
    public class TextToken : Token
    {
        protected string tokenname = string.Empty;
        protected string replacevalue = string.Empty;
        public TextToken(string tokenname, string replacevalue)
        {
            this.tokenname = "%" + tokenname;
            this.replacevalue = replacevalue;
        }
        public string ReplaceToken(string input)
        {
            return input.Replace(this.tokenname, this.replacevalue);
        }
    }
}
